const SERVER_URL = "https://api.mtvoc.com";

function getDeviceId() {
  return new Promise((resolve) => {
    chrome.storage.local.get(["device_id"], (result) => {
      if (result.device_id) {
        resolve(result.device_id);
      } else {
        const newId = crypto.randomUUID();
        chrome.storage.local.set({ device_id: newId }, () => {
          resolve(newId);
        });
      }
    });
  });
}

// ================= View Control =================
function setupTabs() {
  const tabs = document.querySelectorAll(".tab-btn");
  const contents = document.querySelectorAll(".tab-content");

  tabs.forEach((tab) => {
    tab.addEventListener("click", () => {
      // Remove active from all
      tabs.forEach((t) => t.classList.remove("active"));
      contents.forEach((c) => c.classList.remove("active"));

      // Add active to current
      tab.classList.add("active");
      const targetId = `tab-${tab.getAttribute("data-tab")}`;
      const targetContent = document.getElementById(targetId);
      if (targetContent) {
        targetContent.classList.add("active");
      }
    });
  });
}

function showWelcome() {
  document.getElementById("welcome-card").classList.remove("hidden");
  document.getElementById("settings-card").classList.add("hidden");
}

function showSettings() {
  document.getElementById("welcome-card").classList.add("hidden");
  document.getElementById("settings-card").classList.remove("hidden");
  loadSettings();
}

document.addEventListener("DOMContentLoaded", async () => {
  // 检查是否已完成引导
  const { onboarding_complete } = await chrome.storage.local.get([
    "onboarding_complete",
  ]);

  if (onboarding_complete) {
    showSettings();
  } else {
    showWelcome();
  }

  // 初始化设置页面的监听器
  setupSettings();
  setupTabs();

  // Initialize welcome page language buttons
  const langButtons = document.querySelectorAll(".lang-btn");
  langButtons.forEach((button) => {
    button.addEventListener("click", () => {
      const lang = button.getAttribute("data-lang");
      if (lang) {
        window.selectWelcomeLang(lang);
      }
    });
  });
});

// ================= Welcome Page Logic =================
const startBtn = document.getElementById("start-btn");
if (startBtn) {
  startBtn.addEventListener("click", async () => {
    const btn = document.getElementById("start-btn");
    const statusMsg = document.getElementById("status-msg");

    btn.disabled = true;
    btn.textContent = I18N[CURRENT_LANG]["btn_processing"];

    try {
      const deviceId = await getDeviceId();
      const response = await fetch(`${SERVER_URL}/user/start_trial`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ device_id: deviceId }),
      });

      if (response.ok) {
        btn.textContent = I18N[CURRENT_LANG]["btn_success"];
        btn.style.backgroundColor = "#27ae60"; // Green
        statusMsg.textContent = I18N[CURRENT_LANG]["redirect_msg"];

        // 标记引导完成
        await chrome.storage.local.set({ onboarding_complete: true });

        // 延迟跳转，让用户看到成功提示
        setTimeout(() => {
          // 激活成功后，打开演示页面供用户体验
          chrome.tabs.create(
            {
              url: `https://api.mtvoc.com/guide.html?lang=${encodeURIComponent(CURRENT_LANG)}&device_id=${encodeURIComponent(deviceId)}`,
            },
            () => {
              // 关闭当前页面（引导页）
              chrome.tabs.getCurrent((tab) => {
                if (tab) {
                  chrome.tabs.remove(tab.id);
                } else {
                  window.close();
                }
              });
            },
          );
        }, 1500);
      } else {
        throw new Error("Server error");
      }
    } catch (e) {
      console.error(e);
      btn.textContent = I18N[CURRENT_LANG]["btn_retry"];
      btn.disabled = false;
      statusMsg.textContent = I18N[CURRENT_LANG]["error_connect"];
    }
  });
}

// ================= Settings Page Logic =================
const I18N = {
  "zh-CN": {
    language_settings: "语言设置 / Language",
    model_settings: "AI 模型 / AI Model",
    model_v25: "DeepSeek V2.5 (默认)",
    model_v32: "DeepSeek V3.2 (Pro 专用)",
    detail_level_title: "回复详细程度",
    detail_auto_label: "<strong>自动</strong> &nbsp; (根据划线内容自动判断)",
    detail_concise_label: "<strong>精简</strong> &nbsp; (50字以内，直击要点)",
    detail_detailed_label:
      "<strong>详细</strong> &nbsp; (包含背景、原理与扩展信息)",
    membership_title: "会员订阅",
    upgrade_btn: "升级 Pro / 续费",
    upgrade_desc: "对比不同套餐，选择最适合您的方案",
    feat_unlimited_title: "无限查询",
    feat_unlimited_desc: "告别额度限制",
    feat_deep_title: "深度思考",
    feat_deep_desc: "解锁最强模型",
    feat_fast_title: "极速响应",
    feat_fast_desc: "优先处理请求",
    feat_support_title: "售后保障",
    feat_support_desc: "专属客服通道",
    payment_note: "支付完成后，权限通常在 1-3 分钟内自动到账。",
    recover_title: "找回权益",
    recover_desc: "如果重装插件导致会员丢失，请输入订单号找回：",
    recover_btn: "输入订单号找回",
    status_pro:
      "<strong>✨ PRO 会员</strong><br><span style='font-size:0.9em'>无限畅聊 | 深度追问</span><br><span style='font-size:0.8em; opacity:0.8'>有效期至: {date}</span>",
    status_trial:
      "<strong>⚡ 全功能试用中</strong><br><span style='font-size:0.9em'>深度追问剩余: {count} / 20 次</span><br><span style='font-size:0.8em; opacity:0.8'>有效期至: {date}</span>",
    status_free:
      "<strong>免费版</strong><br><span style='font-size:0.9em'>今日剩余查询: {count} / 3 次</span>",
    toast_saved: "✅ 设置已保存",
    page_title: "ExplainAI 设置",
    status_title: "当前状态",
    welcome_title: "恭喜！ExplainAI 已安装就绪",
    welcome_subtitle: "您的阅读效率即将提升 10 倍",
    privilege_title: "您的专属特权已送达",
    privilege_trial: "<strong>48小时</strong> 全功能试用",
    privilege_model: "<strong>深度思考</strong> 模型解锁",
    privilege_quota: "<strong>20次</strong> 深度追问额度",
    privilege_speed: "<strong>零等待</strong> 极速响应",
    pro_ribbon: "PRO 体验卡",
    btn_activate: "立即激活 Pro 体验",
    activate_hint: "试用结束后，您可以选择升级 Pro 会员",
    recover_success: "✅ 权益找回成功！请刷新页面查看状态。",
    recover_fail: "找回失败，请检查订单号",
    recover_error: "❌ 连接服务器失败",
    recover_validating: "正在验证...",
    recover_input_hint: "请输入订单号",
    btn_cancel: "取消",
    btn_confirm: "确认找回",
    step_detail_title: "选择回复详细程度",
    step_detail_subtitle: "您可以随时在设置中更改",
    btn_next: "下一步",
    btn_processing: "正在开启...",
    btn_success: "开启成功！",
    btn_retry: "重试",
    error_connect: "连接服务器失败，请检查网络或后台服务是否运行。",
    privilege_ribbon: "Pro 体验卡",
    redirect_msg: "即将跳转到设置页面...",
    btn_redeem: "🎁 使用兑换码",
    redeem_modal_title: "兑换 Pro 权益",
    redeem_modal_desc: "请输入兑换码",
    placeholder_redeem_code: "请输入兑换码...",
    btn_redeem_confirm: "确认兑换",
    redeem_success: "成功！已兑换 Pro 权益",
    redeem_fail: "兑换失败",
    redeem_validating: "正在兑换...",
    redeem_error: "网络错误或无法连接服务器",
    redeem_input_hint: "请输入有效的兑换码",
    CODE_NOT_FOUND: "兑换码不存在",
    CODE_USED: "兑换码已被使用",
    CODE_INVALID_TYPE: "兑换码类型无效",
    CODE_INVALID_COUNT: "生成数量无效",
    CODE_INVALID_REQUEST: "请求参数无效",
    contact_support: "联系客服",
    interaction_title: "交互方式",
    disable_floating_label: "关闭悬浮窗功能",
    hotkey_mode_label: "选择后不弹出，按快捷键解析",
    hotkey_label: "快捷键",
    hotkey_placeholder: "点击后按下组合键",
    hotkey_hint: "开启后点击输入框，再按下组合键（默认 Shift+Q）",
    hotkey_capture: "请按下组合键… (Esc 取消)",
  },
  en: {
    tab_dashboard: "Dashboard",
    tab_settings: "Settings",
    language_settings: "Language",
    model_settings: "AI Model",
    model_v25: "DeepSeek V2.5 (Default)",
    model_v32: "DeepSeek V3.2 (Pro Only)",
    detail_level_title: "Response Detail",
    detail_auto_label: "<strong>Auto</strong> (Default)",
    detail_concise_label: "<strong>Concise</strong> (<50 words)",
    detail_detailed_label: "<strong>Detailed</strong> (In-depth)",
    membership_title: "Membership",
    upgrade_btn: "Upgrade Pro / Renew",
    upgrade_desc: "Choose the plan that suits you best",
    feat_unlimited_title: "Unlimited",
    feat_unlimited_desc: "No daily limits",
    feat_deep_title: "Deep Think",
    feat_deep_desc: "Unlock best models",
    feat_fast_title: "Fast Response",
    feat_fast_desc: "Priority processing",
    feat_support_title: "Support",
    feat_support_desc: "Dedicated channel",
    payment_note: "Permissions usually activate within 1-3 mins after payment.",
    recover_title: "Restore Purchase",
    recover_desc: "If you lost access after reinstalling, enter Order ID:",
    recover_btn: "Enter Order ID",
    status_pro:
      "<strong>✨ PRO Member</strong><br><span style='font-size:0.9em'>Unlimited | Deep Follow-up</span><br><span style='font-size:0.8em; opacity:0.8'>Valid until: {date}</span>",
    status_trial:
      "<strong>⚡ Full Trial</strong><br><span style='font-size:0.9em'>Follow-ups left: {count} / 20</span><br><span style='font-size:0.8em; opacity:0.8'>Valid until: {date}</span>",
    status_free:
      "<strong>Free Version</strong><br><span style='font-size:0.9em'>Daily Queries: {count} / 3 left</span>",
    toast_saved: "✅ Settings Saved",
    page_title: "ExplainAI Settings",
    status_title: "Current Status",
    welcome_title: "Congratulations! ExplainAI is Ready",
    welcome_subtitle: "Your reading efficiency is about to boost 10x",
    privilege_title: "Your Exclusive Privileges",
    privilege_trial: "<strong>48-Hour</strong> Full Trial",
    privilege_model: "<strong>Deep Think</strong> Model",
    privilege_quota: "<strong>20</strong> Deep Queries",
    privilege_speed: "<strong>Zero Wait</strong> Fast Response",
    btn_activate: "Activate Pro Trial Now",
    activate_hint: "You can upgrade to Pro after trial ends",
    recover_success: "✅ Restored successfully! Please refresh.",
    recover_fail: "Restore failed. Check Order ID.",
    recover_error: "❌ Connection failed",
    recover_validating: "Validating...",
    recover_input_hint: "Please enter Order ID",
    btn_cancel: "Cancel",
    btn_confirm: "Confirm",
    pro_ribbon: "PRO Trial Card",
    redirect_msg: "Redirecting to settings...",
    step_detail_title: "Select Response Detail",
    step_detail_subtitle: "You can change this anytime in settings",
    btn_next: "Next",
    btn_processing: "Processing...",
    btn_success: "Success!",
    btn_retry: "Retry",
    error_connect: "Connection failed, please check network.",
    btn_redeem: "🎁 Redeem code",
    redeem_modal_title: "Redeem Pro",
    redeem_modal_desc: "Enter your redemption code",
    placeholder_redeem_code: "Enter code...",
    btn_redeem_confirm: "Redeem",
    redeem_success: "Success! Pro redeemed.",
    redeem_fail: "Redeem failed",
    redeem_validating: "Redeeming...",
    redeem_error: "Network error or server unreachable",
    redeem_input_hint: "Please enter a valid code",
    CODE_NOT_FOUND: "Code not found",
    CODE_USED: "Code already used",
    CODE_INVALID_TYPE: "Invalid code type",
    CODE_INVALID_COUNT: "Invalid count",
    CODE_INVALID_REQUEST: "Invalid request",
    contact_support: "Contact Support",
    interaction_title: "Interaction",
    disable_floating_label: "Disable floating popup",
    hotkey_mode_label: "No popup on selection, use hotkey to explain",
    hotkey_label: "Hotkey",
    hotkey_placeholder: "Click to set hotkey",
    hotkey_hint: "Enable it, click input, then press keys (default Shift+Q)",
    hotkey_capture: "Press keys… (Esc to cancel)",
  },
};

let CURRENT_LANG = "zh-CN";

// Make this global so it can be called from HTML
window.selectWelcomeLang = function (lang) {
  CURRENT_LANG = lang;
  chrome.storage.local.set({ language: lang }, async () => {
    updateTexts();
    // Switch to step 1.5 (Detail Selection)
    document.getElementById("welcome-step-1").classList.add("hidden");
    document.getElementById("welcome-step-detail").classList.remove("hidden");

    // Sync to backend
    try {
      const deviceId = await getDeviceId();
      await fetch(`${SERVER_URL}/user/update_profile`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          device_id: deviceId,
          language: lang,
        }),
      });
    } catch (err) {
      console.error("Failed to sync language", err);
    }
  });
};

document.addEventListener("DOMContentLoaded", () => {
  // Setup Detail Step Next Button
  const btnDetailNext = document.getElementById("btn-detail-next");
  if (btnDetailNext) {
    btnDetailNext.addEventListener("click", () => {
      const selectedDetail = document.querySelector(
        'input[name="welcome_detail"]:checked',
      ).value;
      chrome.storage.local.set({ detail_level: selectedDetail }, () => {
        // Switch to step 2 (Activation)
        document.getElementById("welcome-step-detail").classList.add("hidden");
        document.getElementById("welcome-step-2").classList.remove("hidden");
      });
    });
  }
});

function updateTexts() {
  const texts = I18N[CURRENT_LANG];

  // Update Title
  document.title = texts["page_title"];

  document.querySelectorAll("[data-i18n]").forEach((el) => {
    const key = el.getAttribute("data-i18n");
    if (texts[key]) {
      // Check if it has child elements (like strong)
      if (
        (el.children.length > 0 || texts[key].includes("<")) &&
        (key.includes("detail_") ||
          key.includes("upgrade_btn") ||
          key.includes("privilege_") ||
          key.includes("status_") ||
          key.includes("pro_ribbon"))
      ) {
        el.innerHTML = texts[key];
      } else {
        el.textContent = texts[key];
      }
    }
  });

  document.querySelectorAll("[data-i18n-placeholder]").forEach((el) => {
    const key = el.getAttribute("data-i18n-placeholder");
    if (texts[key]) {
      el.placeholder = texts[key];
    }
  });
}

async function loadSettings() {
  const deviceId = await getDeviceId();
  const deviceIdDisplay = document.getElementById("device-id-display");
  if (deviceIdDisplay) deviceIdDisplay.textContent = deviceId;

  // 0. 加载语言设置
  const { language } = await chrome.storage.local.get(["language"]);
  if (language) {
    CURRENT_LANG = language;
    const radio = document.querySelector(
      `input[name="language"][value="${language}"]`,
    );
    if (radio) radio.checked = true;
  } else {
    // Detect system language
    const sysLang = navigator.language.startsWith("zh") ? "zh-CN" : "en";
    CURRENT_LANG = sysLang;
    const defaultRadio = document.querySelector(
      `input[name="language"][value="${sysLang}"]`,
    );
    if (defaultRadio) defaultRadio.checked = true;
    chrome.storage.local.set({ language: sysLang });
  }
  updateTexts();

  const pricingLink = document.getElementById("open-pricing-link");
  if (pricingLink) {
    const params = new URLSearchParams({
      device_id: deviceId,
      lang: CURRENT_LANG,
    });
    pricingLink.href = `${SERVER_URL}/pricing?${params.toString()}`;
  }

  const { disable_floating_window, hotkey_mode_enabled, hotkey_combo } =
    await chrome.storage.local.get([
      "disable_floating_window",
      "hotkey_mode_enabled",
      "hotkey_combo",
    ]);

  const disableFloatingEl = document.getElementById(
    "disable-floating-checkbox",
  );
  if (disableFloatingEl) {
    disableFloatingEl.checked = Boolean(disable_floating_window);
  }

  const hotkeyModeEl = document.getElementById("hotkey-mode-checkbox");
  const hotkeyInput = document.getElementById("hotkey-input");

  const defaultHotkey = {
    code: "KeyQ",
    shift: true,
    ctrl: false,
    alt: false,
    meta: false,
  };

  const hk =
    hotkey_combo && typeof hotkey_combo === "object"
      ? hotkey_combo
      : defaultHotkey;

  if (!hotkey_combo) {
    chrome.storage.local.set({ hotkey_combo: defaultHotkey });
  }

  if (hotkeyModeEl) {
    hotkeyModeEl.checked = Boolean(hotkey_mode_enabled);
  } else if (hotkey_mode_enabled === undefined) {
    chrome.storage.local.set({ hotkey_mode_enabled: false });
  }

  if (disable_floating_window === undefined) {
    chrome.storage.local.set({ disable_floating_window: false });
  }

  if (hotkeyInput) {
    hotkeyInput.value = formatHotkey(hk);
    setHotkeyInputEnabled(hotkeyInput, Boolean(hotkey_mode_enabled));
  }

  // 1. 加载详细程度设置
  const { detail_level } = await chrome.storage.local.get(["detail_level"]);

  if (detail_level) {
    const radio = document.querySelector(
      `input[name="detail_level"][value="${detail_level}"]`,
    );
    if (radio) radio.checked = true;
  } else {
    // 默认选中自动
    const defaultRadio = document.querySelector(
      `input[name="detail_level"][value="auto"]`,
    );
    if (defaultRadio) defaultRadio.checked = true;
    // 保存默认设置
    chrome.storage.local.set({ detail_level: "auto" });
  }

  // 1.5 Load AI Model
  const { ai_model } = await chrome.storage.local.get(["ai_model"]);
  if (ai_model) {
    const radio = document.querySelector(
      `input[name="ai_model"][value="${ai_model}"]`,
    );
    if (radio) radio.checked = true;
  } else {
    chrome.storage.local.set({ ai_model: "deepseek-ai/DeepSeek-V2.5" });
  }

  // 2. 加载用户状态
  const statusDisplay = document.getElementById("user-status-display");
  if (statusDisplay) {
    try {
      const res = await fetch(
        `${SERVER_URL}/user/status?device_id=${deviceId}&_=${Date.now()}`,
      );
      const data = await res.json();

      const modelV3Input = document.querySelector(
        'input[name="ai_model"][value="deepseek-ai/DeepSeek-V3.2"]',
      );
      const modelV3Label = document.getElementById("model-v3-label");

      if (data.is_pro) {
        if (modelV3Input) modelV3Input.disabled = false;
        if (modelV3Label) modelV3Label.classList.remove("disabled-option");

        const expireDate = data.expire_at ? new Date(data.expire_at) : null;
        const isLifetime = expireDate && expireDate.getFullYear() > 2090;
        const expireText = isLifetime
          ? CURRENT_LANG === "en"
            ? "Lifetime"
            : "永久有效"
          : expireDate
            ? expireDate.toLocaleDateString()
            : CURRENT_LANG === "en"
              ? "Unknown"
              : "未知";

        statusDisplay.innerHTML = I18N[CURRENT_LANG]["status_pro"].replace(
          "{date}",
          expireText,
        );
        statusDisplay.style.background = "#e8f8f5";
        statusDisplay.style.color = "#27ae60";
        statusDisplay.style.borderColor = "#d4efdf";
      } else if (data.trial_active) {
        if (modelV3Input) {
          modelV3Input.disabled = true;
          if (modelV3Input.checked) {
            const defaultRadio = document.querySelector(
              'input[name="ai_model"][value="deepseek-ai/DeepSeek-V2.5"]',
            );
            if (defaultRadio) {
              defaultRadio.checked = true;
              chrome.storage.local.set({
                ai_model: "deepseek-ai/DeepSeek-V2.5",
              });
            }
          }
        }
        if (modelV3Label) modelV3Label.classList.add("disabled-option");

        const trialEnd = data.trial_expire_date
          ? new Date(data.trial_expire_date)
          : new Date(
              new Date(data.trial_start_date).getTime() + 48 * 3600 * 1000,
            );
        statusDisplay.innerHTML = I18N[CURRENT_LANG]["status_trial"]
          .replace("{count}", 20 - data.followup_count)
          .replace("{date}", trialEnd.toLocaleString());
        statusDisplay.style.background = "#fff8e1";
        statusDisplay.style.color = "#d35400";
        statusDisplay.style.borderColor = "#fce8b2";
      } else {
        if (modelV3Input) {
          modelV3Input.disabled = true;
          if (modelV3Input.checked) {
            const defaultRadio = document.querySelector(
              'input[name="ai_model"][value="deepseek-ai/DeepSeek-V2.5"]',
            );
            if (defaultRadio) {
              defaultRadio.checked = true;
              chrome.storage.local.set({
                ai_model: "deepseek-ai/DeepSeek-V2.5",
              });
            }
          }
        }
        if (modelV3Label) modelV3Label.classList.add("disabled-option");

        statusDisplay.innerHTML = I18N[CURRENT_LANG]["status_free"].replace(
          "{count}",
          3 - data.daily_query_count,
        );
        statusDisplay.style.background = "#f2f3f4";
        statusDisplay.style.color = "#7f8c8d";
        statusDisplay.style.borderColor = "#bdc3c7";
      }
    } catch (e) {
      statusDisplay.textContent =
        CURRENT_LANG === "en"
          ? "Status Check Failed"
          : "状态检查失败 (无法连接服务器)";
      statusDisplay.style.background = "#fdedec";
      statusDisplay.style.color = "#c0392b";
      statusDisplay.style.borderColor = "#fadbd8";
    }
  }
}

function showToast(message) {
  let toast = document.getElementById("toast-msg");
  if (!toast) {
    toast = document.createElement("div");
    toast.id = "toast-msg";
    toast.style.position = "fixed";
    toast.style.bottom = "30px";
    toast.style.left = "50%";
    toast.style.transform = "translateX(-50%)";
    toast.style.backgroundColor = "rgba(46, 204, 113, 0.9)";
    toast.style.color = "white";
    toast.style.padding = "12px 24px";
    toast.style.borderRadius = "30px";
    toast.style.zIndex = "1000";
    toast.style.boxShadow = "0 4px 12px rgba(0,0,0,0.15)";
    toast.style.transition = "all 0.3s ease";
    toast.style.opacity = "0";
    toast.style.fontWeight = "bold";
    document.body.appendChild(toast);
  }

  // Trigger reflow
  void toast.offsetWidth;

  toast.textContent = message;
  toast.style.opacity = "1";
  toast.style.transform = "translateX(-50%) translateY(0)";

  setTimeout(() => {
    toast.style.opacity = "0";
    toast.style.transform = "translateX(-50%) translateY(10px)";
  }, 2000);
}

function isModifierCode(code) {
  return (
    code === "ShiftLeft" ||
    code === "ShiftRight" ||
    code === "ControlLeft" ||
    code === "ControlRight" ||
    code === "AltLeft" ||
    code === "AltRight" ||
    code === "MetaLeft" ||
    code === "MetaRight"
  );
}

function codeToKeyLabel(code) {
  if (!code) return "";
  if (code.startsWith("Key")) return code.slice(3);
  if (code.startsWith("Digit")) return code.slice(5);
  return code;
}

function formatHotkey(hk) {
  const parts = [];
  if (hk && hk.ctrl) parts.push("Ctrl");
  if (hk && hk.alt) parts.push("Alt");
  if (hk && hk.shift) parts.push("Shift");
  if (hk && hk.meta) parts.push("Meta");
  parts.push(codeToKeyLabel(hk && hk.code ? hk.code : ""));
  return parts.filter(Boolean).join("+");
}

function setHotkeyInputEnabled(inputEl, enabled) {
  if (!inputEl) return;
  if (enabled) {
    inputEl.classList.remove("disabled");
    inputEl.dataset.disabled = "false";
  } else {
    inputEl.classList.add("disabled");
    inputEl.dataset.disabled = "true";
  }
}

function setupSettings() {
  const modelRadios = document.querySelectorAll('input[name="ai_model"]');
  modelRadios.forEach((radio) => {
    radio.addEventListener("change", (e) => {
      chrome.storage.local.set({ ai_model: e.target.value }, () => {
        showToast(I18N[CURRENT_LANG]["toast_saved"]);
      });
    });
  });

  const radios = document.querySelectorAll('input[name="detail_level"]');
  radios.forEach((radio) => {
    radio.addEventListener("change", (e) => {
      chrome.storage.local.set({ detail_level: e.target.value }, () => {
        showToast(I18N[CURRENT_LANG]["toast_saved"]);
      });
    });
  });

  const langRadios = document.querySelectorAll('input[name="language"]');
  langRadios.forEach((radio) => {
    radio.addEventListener("change", (e) => {
      const newLang = e.target.value;
      CURRENT_LANG = newLang;
      chrome.storage.local.set({ language: newLang }, async () => {
        updateTexts();
        loadSettings(); // Reload status to update language
        showToast(I18N[CURRENT_LANG]["toast_saved"]);

        // Sync to backend
        try {
          const deviceId = await getDeviceId();
          await fetch(`${SERVER_URL}/user/update_profile`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({
              device_id: deviceId,
              language: newLang,
            }),
          });
        } catch (err) {
          console.error("Failed to sync language", err);
        }
      });
    });
  });

  const disableFloatingEl = document.getElementById(
    "disable-floating-checkbox",
  );
  if (disableFloatingEl) {
    disableFloatingEl.addEventListener("change", (e) => {
      chrome.storage.local.set(
        { disable_floating_window: Boolean(e.target.checked) },
        () => {
          showToast(I18N[CURRENT_LANG]["toast_saved"]);
        },
      );
    });
  }

  const hotkeyModeEl = document.getElementById("hotkey-mode-checkbox");
  const hotkeyInput = document.getElementById("hotkey-input");
  if (hotkeyModeEl) {
    hotkeyModeEl.addEventListener("change", (e) => {
      const enabled = Boolean(e.target.checked);
      chrome.storage.local.set({ hotkey_mode_enabled: enabled }, () => {
        if (hotkeyInput) setHotkeyInputEnabled(hotkeyInput, enabled);
        showToast(I18N[CURRENT_LANG]["toast_saved"]);
      });
    });
  }

  if (hotkeyInput) {
    let capturing = false;
    let lastValue = hotkeyInput.value;

    hotkeyInput.addEventListener("click", () => {
      if (hotkeyInput.dataset.disabled === "true") return;
      capturing = true;
      lastValue = hotkeyInput.value;
      hotkeyInput.value = I18N[CURRENT_LANG]["hotkey_capture"];
      hotkeyInput.classList.add("capturing");
      hotkeyInput.focus();
    });

    document.addEventListener("keydown", (e) => {
      if (!capturing) return;
      e.preventDefault();
      e.stopPropagation();

      if (e.key === "Escape") {
        capturing = false;
        hotkeyInput.classList.remove("capturing");
        hotkeyInput.value = lastValue;
        return;
      }

      if (isModifierCode(e.code)) return;

      const hk = {
        code: e.code,
        shift: e.shiftKey,
        ctrl: e.ctrlKey,
        alt: e.altKey,
        meta: e.metaKey,
      };

      const display = formatHotkey(hk);
      chrome.storage.local.set({ hotkey_combo: hk }, () => {
        capturing = false;
        hotkeyInput.classList.remove("capturing");
        hotkeyInput.value = display;
        showToast(I18N[CURRENT_LANG]["toast_saved"]);
      });
    });
  }

  const recoverBtn = document.getElementById("open-recover-modal");
  const recoverModal = document.getElementById("recover-modal");
  const cancelRecoverBtn = document.getElementById("cancel-recover");
  const confirmRecoverBtn = document.getElementById("confirm-recover");

  if (recoverBtn && recoverModal) {
    recoverBtn.addEventListener("click", () => {
      recoverModal.style.display = "flex";
      document.getElementById("recover-msg-modal").textContent = "";
      document.getElementById("recover-order-id-modal").value = "";
    });
  }

  if (cancelRecoverBtn && recoverModal) {
    cancelRecoverBtn.addEventListener("click", () => {
      recoverModal.style.display = "none";
    });
  }

  if (confirmRecoverBtn) {
    confirmRecoverBtn.addEventListener("click", recoverPurchase);
  }

  // Close modal when clicking outside
  if (recoverModal) {
    recoverModal.addEventListener("click", (e) => {
      if (e.target === recoverModal) {
        recoverModal.style.display = "none";
      }
    });
  }

  // Redeem Modal Logic
  const redeemModal = document.getElementById("redeem-modal");
  const openRedeemBtn = document.getElementById("open-redeem-modal");
  const cancelRedeemBtn = document.getElementById("cancel-redeem");
  const confirmRedeemBtn = document.getElementById("confirm-redeem");
  const redeemMsgEl = document.getElementById("redeem-msg-modal");
  const redeemInputEl = document.getElementById("redeem-code-modal");

  if (openRedeemBtn && redeemModal) {
    openRedeemBtn.addEventListener("click", () => {
      redeemModal.style.display = "flex";
      if (redeemInputEl) {
        redeemInputEl.value = "";
        redeemInputEl.focus();
      }
      if (redeemMsgEl) redeemMsgEl.textContent = "";
    });

    redeemModal.addEventListener("click", (e) => {
      if (e.target === redeemModal) {
        redeemModal.style.display = "none";
      }
    });
  }

  if (cancelRedeemBtn && redeemModal) {
    cancelRedeemBtn.addEventListener("click", () => {
      redeemModal.style.display = "none";
    });
  }

  if (confirmRedeemBtn && redeemInputEl && redeemMsgEl) {
    confirmRedeemBtn.addEventListener("click", async () => {
      const code = redeemInputEl.value.trim();

      if (!code) {
        redeemMsgEl.textContent = I18N[CURRENT_LANG]["redeem_input_hint"];
        redeemMsgEl.style.color = "red";
        return;
      }

      redeemMsgEl.textContent = I18N[CURRENT_LANG]["redeem_validating"];
      redeemMsgEl.style.color = "#666";
      confirmRedeemBtn.disabled = true;

      try {
        const deviceId = await getDeviceId();
        const res = await fetch(`${SERVER_URL}/redeem`, {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            device_id: deviceId,
            code: code,
          }),
        });

        const data = await res.json();

        if (res.ok) {
          redeemMsgEl.textContent =
            "✅ " + I18N[CURRENT_LANG]["redeem_success"];
          redeemMsgEl.style.color = "green";
          loadSettings(); // Refresh status
          setTimeout(() => {
            redeemModal.style.display = "none";
            confirmRedeemBtn.disabled = false;
          }, 1500);
        } else {
          const errorMsg = data.detail || data.message || "redeem_fail";
          redeemMsgEl.textContent =
            "❌ " + (I18N[CURRENT_LANG][errorMsg] || errorMsg);
          redeemMsgEl.style.color = "red";
          confirmRedeemBtn.disabled = false;
        }
      } catch (e) {
        console.error(e);
        redeemMsgEl.textContent = "❌ " + I18N[CURRENT_LANG]["redeem_error"];
        redeemMsgEl.style.color = "red";
        confirmRedeemBtn.disabled = false;
      }
    });
  }
}

// ================= Payment & Recovery =================

async function recoverPurchase() {
  const orderIdInput = document.getElementById("recover-order-id-modal");
  const msg = document.getElementById("recover-msg-modal");
  const outTradeNo = orderIdInput.value.trim();

  if (!outTradeNo) {
    msg.textContent = I18N[CURRENT_LANG]["recover_input_hint"];
    msg.style.color = "red";
    return;
  }

  msg.textContent = I18N[CURRENT_LANG]["recover_validating"];
  msg.style.color = "#666";

  try {
    const deviceId = await getDeviceId();
    const response = await fetch(`${SERVER_URL}/recover_pro`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        device_id: deviceId,
        out_trade_no: outTradeNo,
      }),
    });

    const data = await response.json();

    if (response.ok) {
      msg.textContent = I18N[CURRENT_LANG]["recover_success"];
      msg.style.color = "green";
      loadSettings(); // 刷新状态
    } else {
      let errorMsg = I18N[CURRENT_LANG]["recover_fail"];
      if (data.detail === "ORDER_NOT_FOUND") {
        errorMsg = I18N[CURRENT_LANG]["recover_fail"];
      } else if (data.message) {
        errorMsg = data.message;
      } else if (data.detail) {
        errorMsg = data.detail;
      }
      msg.textContent = `❌ ${errorMsg}`;
      msg.style.color = "red";
    }
  } catch (e) {
    msg.textContent = I18N[CURRENT_LANG]["recover_error"];
    msg.style.color = "red";
    console.error(e);
  }
}
