// ================= 配置 =================
const API_BASE = "https://api.mtvoc.com";

const I18N = {
  "zh-CN": {
    ask_ai: "Ask AI",
    panel_title: "ExplainIt AI",
    placeholder_ask: "追问...",
    footer_checking: "Checking status...",
    status_pro: "✨ PRO 会员 | 无限畅聊",
    status_trial_prefix: "⚡ 全功能试用中 (深度追问 {count}/20)",
    status_free_prefix: "免费版 ({count}/3)",
    status_offline_prefix: "离线",
    status_upgrade_trial: "去升级",
    status_upgrade_free: "升级 Pro",
    status_upgrade_offline: "升级 Pro (离线)",
    trial_modal_title: "您的试用已结束",
    trial_modal_desc:
      "在过去的 48 小时里，你通过 ExplainIt 节省了约 {minutes} 分钟的时间。",
    trial_modal_info:
      "试用已结束。你现在可以继续使用“每日 3 次”的基础划线功能，但您将无法使用“深度追问”功能。",
    trial_modal_pro: "如果您想继续使用全部功能，您可以选择成为我们的Pro会员。",
    trial_modal_cta: "别让繁琐的搜索打断宝贵的心流状态。",
    trial_modal_btn: "升级至 Pro 版（￥68/季，节省 25%）",
    trial_modal_quote: "“在深度追问中不断进化”",
    error_unknown: "未知服务端错误",
    error_network: "请检查网络连接。",
    error_check_network: "请检查网络或代理设置。",
  },
  en: {
    ask_ai: "Ask AI",
    panel_title: "ExplainIt AI",
    placeholder_ask: "Follow up...",
    footer_checking: "Checking status...",
    status_pro: "✨ PRO Member | Unlimited",
    status_trial_prefix: "⚡ Full Trial (Follow-ups {count}/20)",
    status_free_prefix: "Free Plan ({count}/3)",
    status_offline_prefix: "Offline",
    status_upgrade_trial: "Upgrade",
    status_upgrade_free: "Upgrade Pro",
    status_upgrade_offline: "Upgrade Pro (Offline)",
    trial_modal_title: "Trial Ended",
    trial_modal_desc:
      "In the past 48 hours, ExplainIt saved you about {minutes} minutes.",
    trial_modal_info:
      "Trial ended. You can continue with 3 daily queries, but 'Deep Follow-up' is disabled.",
    trial_modal_pro:
      "To continue using all features, consider upgrading to Pro.",
    trial_modal_cta: "Don't let search interruptions break your flow.",
    trial_modal_btn: "Upgrade to Pro",
    trial_modal_quote: "“Evolve through deep inquiry”",
    error_unknown: "Unknown Server Error",
    error_network: "Check network connection.",
    error_check_network: "Check network or proxy settings.",
  },
};

let CURRENT_LANG = "zh-CN";
let DISABLE_FLOATING_WINDOW = false;
let HOTKEY_MODE_ENABLED = false;
let HOTKEY_COMBO = {
  code: "KeyQ",
  shift: true,
  ctrl: false,
  alt: false,
  meta: false,
};
let LAST_SELECTION = { text: "", x: 0, y: 0 };

// Initialize Language
chrome.storage.local.get(["language"], (res) => {
  if (res.language) {
    CURRENT_LANG = res.language;
  } else {
    // Detect
    CURRENT_LANG = navigator.language.startsWith("zh") ? "zh-CN" : "en";
  }
});

chrome.storage.onChanged.addListener((changes, area) => {
  if (area === "local" && changes.language) {
    CURRENT_LANG = changes.language.newValue;
  }
  if (area === "local" && changes.disable_floating_window) {
    DISABLE_FLOATING_WINDOW = Boolean(changes.disable_floating_window.newValue);
    if (DISABLE_FLOATING_WINDOW) {
      removeIcon();
      removePanel();
    }
  }
  if (area === "local" && changes.hotkey_mode_enabled) {
    HOTKEY_MODE_ENABLED = Boolean(changes.hotkey_mode_enabled.newValue);
    if (HOTKEY_MODE_ENABLED) {
      removeIcon();
    }
  }
  if (area === "local" && changes.hotkey_combo) {
    const next = changes.hotkey_combo.newValue;
    if (next && typeof next === "object") {
      HOTKEY_COMBO = next;
    }
  }
});

function t(key) {
  return I18N[CURRENT_LANG][key] || key;
}

chrome.storage.local.get(
  ["disable_floating_window", "hotkey_mode_enabled", "hotkey_combo"],
  (res) => {
    if (res && res.disable_floating_window !== undefined) {
      DISABLE_FLOATING_WINDOW = Boolean(res.disable_floating_window);
    }
    if (res && res.hotkey_mode_enabled !== undefined) {
      HOTKEY_MODE_ENABLED = Boolean(res.hotkey_mode_enabled);
    }
    if (res && res.hotkey_combo && typeof res.hotkey_combo === "object") {
      HOTKEY_COMBO = res.hotkey_combo;
    }
  },
);

function buildAfdianItemUrl(deviceId) {
  const params = new URLSearchParams({
    device_id: deviceId,
    lang: CURRENT_LANG,
  });
  return `${API_BASE}/pricing?${params.toString()}`;
}

// ================= 初始化 Device ID =================
async function getDeviceId() {
  try {
    // 检查 chrome.storage 是否可用（处理扩展更新后的上下文丢失问题）
    if (
      typeof chrome === "undefined" ||
      !chrome.storage ||
      !chrome.storage.local
    ) {
      throw new Error("Extension context invalidated (storage undefined)");
    }

    const result = await chrome.storage.local.get(["device_id"]);
    if (result.device_id) {
      return result.device_id;
    } else {
      // 生成 UUID (兼容非安全上下文，HTTP 下 crypto.randomUUID 不可用)
      let newId;
      if (
        typeof self.crypto !== "undefined" &&
        typeof self.crypto.randomUUID === "function"
      ) {
        newId = self.crypto.randomUUID();
      } else {
        newId = "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(
          /[xy]/g,
          function (c) {
            const r = (Math.random() * 16) | 0;
            const v = c === "x" ? r : (r & 0x3) | 0x8;
            return v.toString(16);
          },
        );
      }
      await chrome.storage.local.set({ device_id: newId });
      return newId;
    }
  } catch (e) {
    console.warn("ExplainIt Context Error:", e);
    // 捕获扩展更新导致的上下文丢失错误或 storage 未定义错误
    if (
      e.message.includes("Extension context invalidated") ||
      e.message.includes("storage undefined") ||
      e.message.includes("Cannot read properties of undefined")
    ) {
      alert("ExplainIt 插件已更新，请刷新当前页面以继续使用。");
      return null;
    }
    throw e;
  }
}

function clearElement(el) {
  while (el.firstChild) el.removeChild(el.firstChild);
}

function renderInline(parent, text) {
  let i = 0;
  while (i < text.length) {
    const nextCode = text.indexOf("`", i);
    const nextBold = text.indexOf("**", i);

    let nextToken = -1;
    let tokenType = null;
    if (nextCode !== -1) {
      nextToken = nextCode;
      tokenType = "code";
    }
    if (nextBold !== -1 && (nextToken === -1 || nextBold < nextToken)) {
      nextToken = nextBold;
      tokenType = "bold";
    }

    if (nextToken === -1) {
      parent.appendChild(document.createTextNode(text.slice(i)));
      return;
    }

    if (nextToken > i) {
      parent.appendChild(document.createTextNode(text.slice(i, nextToken)));
    }

    if (tokenType === "code") {
      const end = text.indexOf("`", nextToken + 1);
      if (end === -1) {
        parent.appendChild(document.createTextNode(text.slice(nextToken)));
        return;
      }
      const codeEl = document.createElement("code");
      codeEl.textContent = text.slice(nextToken + 1, end);
      parent.appendChild(codeEl);
      i = end + 1;
      continue;
    }

    if (tokenType === "bold") {
      const end = text.indexOf("**", nextToken + 2);
      if (end === -1) {
        parent.appendChild(document.createTextNode(text.slice(nextToken)));
        return;
      }
      const strongEl = document.createElement("strong");
      strongEl.textContent = text.slice(nextToken + 2, end);
      parent.appendChild(strongEl);
      i = end + 2;
      continue;
    }

    parent.appendChild(document.createTextNode(text.slice(nextToken)));
    return;
  }
}

function markdownToFragment(text) {
  const frag = document.createDocumentFragment();
  const lines = String(text || "")
    .replace(/\r\n/g, "\n")
    .split("\n");

  let inCode = false;
  let codeLines = [];
  let currentList = null;

  const flushList = () => {
    if (currentList) {
      frag.appendChild(currentList);
      currentList = null;
    }
  };

  const flushCode = () => {
    const pre = document.createElement("pre");
    const code = document.createElement("code");
    code.textContent = codeLines.join("\n");
    pre.appendChild(code);
    frag.appendChild(pre);
    codeLines = [];
  };

  for (const rawLine of lines) {
    const line = rawLine ?? "";

    if (line.trim().startsWith("```")) {
      if (inCode) {
        inCode = false;
        flushList();
        flushCode();
      } else {
        flushList();
        inCode = true;
        codeLines = [];
      }
      continue;
    }

    if (inCode) {
      codeLines.push(line);
      continue;
    }

    if (line.trim() === "") {
      flushList();
      frag.appendChild(document.createElement("br"));
      continue;
    }

    if (line.startsWith("- ")) {
      if (!currentList) currentList = document.createElement("ul");
      const li = document.createElement("li");
      renderInline(li, line.slice(2));
      currentList.appendChild(li);
      continue;
    }

    flushList();

    if (line.startsWith("### ")) {
      const h3 = document.createElement("h3");
      renderInline(h3, line.slice(4));
      frag.appendChild(h3);
      continue;
    }
    if (line.startsWith("## ")) {
      const h2 = document.createElement("h2");
      renderInline(h2, line.slice(3));
      frag.appendChild(h2);
      continue;
    }
    if (line.startsWith("# ")) {
      const h1 = document.createElement("h1");
      renderInline(h1, line.slice(2));
      frag.appendChild(h1);
      continue;
    }

    const p = document.createElement("p");
    renderInline(p, line);
    frag.appendChild(p);
  }

  if (inCode) {
    flushList();
    flushCode();
  } else {
    flushList();
  }

  return frag;
}

function renderMarkdown(container, text) {
  clearElement(container);
  container.appendChild(markdownToFragment(text));
}

// ================= UI 创建工具 =================
function createUI() {
  // 使用 Shadow DOM 隔离样式，防止被宿主网页影响
  const host = document.createElement("div");
  host.id = "explainit-host";
  host.style.position = "absolute";
  host.style.zIndex = "2147483647"; // Max z-index
  host.style.top = "0";
  host.style.left = "0";
  host.style.pointerEvents = "none"; // 默认不阻挡点击，子元素开启
  document.body.appendChild(host);

  const shadow = host.attachShadow({ mode: "open" });

  // 注入基础样式
  const style = document.createElement("style");
  style.textContent = `
        /* Reset & Base Styles */
        :host {
            all: initial; /* 重置所有继承属性 */
            font-family: 'Segoe UI', system-ui, sans-serif;
            font-size: 14px;
            line-height: 1.5;
            color: #333;
            visibility: visible;
        }
        
        div, span, p, a, svg, path, input, button {
            box-sizing: border-box;
        }

        /* 动画关键帧 */
        @keyframes popIn {
            0% { transform: scale(0); opacity: 0; }
            80% { transform: scale(1.1); opacity: 1; }
            100% { transform: scale(1); }
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .icon-btn {
            position: absolute;
            background: linear-gradient(135deg, #3b82f6, #2563eb);
            color: white;
            height: 36px;
            padding: 0 12px;
            border-radius: 18px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 6px;
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.4);
            font-weight: 600;
            font-family: 'Segoe UI', system-ui, sans-serif;
            font-size: 14px;
            white-space: nowrap;
            pointer-events: auto;
            transition: all 0.2s ease;
            z-index: 100;
            animation: popIn 0.3s cubic-bezier(0.175, 0.885, 0.32, 1.275) forwards;
            border: 2px solid white;
        }
        .icon-btn:hover { 
            transform: scale(1.1) translateY(-2px); 
            box-shadow: 0 6px 16px rgba(37, 99, 235, 0.5);
        }
        .icon-svg {
            display: none;
        }
        
        .panel {
            position: absolute;
            width: 340px;
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 8px 10px -6px rgba(0, 0, 0, 0.1);
            display: flex;
            flex-direction: column;
            font-family: 'Segoe UI', system-ui, sans-serif; /* 再次强制指定字体 */
            font-size: 14px;
            line-height: 1.5;
            color: #334155;
            pointer-events: auto;
            overflow: hidden;
            animation: fadeIn 0.2s ease-out forwards;
            z-index: 101;
            text-align: left; /* 防止宿主设置 text-align */
        }
        .header {
            padding: 12px 16px;
            background: linear-gradient(to right, #f8fafc, #f1f5f9);
            border-bottom: 1px solid #e2e8f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-weight: 600;
            color: #1e293b;
            font-size: 15px;
        }
        .close-btn { 
            cursor: pointer; 
            color: #94a3b8; 
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: background 0.2s;
        }
        .close-btn:hover { background: #e2e8f0; color: #64748b; }
        
        .content {
            padding: 16px;
            max-height: 300px; /* 减小高度给输入框腾位置 */
            overflow-y: auto;
            line-height: 1.6;
            color: #334155;
            background: #fff;
        }
        /* 聊天气泡样式 */
        .msg-user {
            text-align: right;
            margin: 8px 0;
        }
        .msg-user-content {
            display: inline-block;
            background: #eff6ff;
            color: #1e40af;
            padding: 6px 12px;
            border-radius: 12px 12px 0 12px;
            font-size: 13px;
            max-width: 85%;
            word-wrap: break-word;
        }
        .msg-ai {
            text-align: left;
            margin: 8px 0;
        }
        .msg-ai-content {
            /* Markdown 样式容器 */
        }

        /* Markdown 样式 */
        .content strong { color: #0f172a; font-weight: 700; }
        .content code { 
            background: #f1f5f9; 
            padding: 2px 4px; 
            border-radius: 4px; 
            color: #ec4899; 
            font-family: monospace;
            font-size: 0.9em;
        }
        .content pre { 
            background: #1e293b; 
            color: #f8fafc; 
            padding: 12px; 
            border-radius: 6px; 
            overflow-x: auto; 
            margin: 8px 0;
        }
        .content pre code {
            background: transparent;
            color: inherit;
            padding: 0;
        }
        .content h1, .content h2, .content h3 { margin: 10px 0 5px; color: #1e293b; }
        .content li { margin-left: 20px; list-style-type: disc; }

        .input-area {
            padding: 10px 16px;
            border-top: 1px solid #e2e8f0;
            background: #fff;
            display: flex;
            gap: 8px;
        }
        .chat-input {
            flex: 1;
            border: 1px solid #cbd5e1;
            border-radius: 20px;
            padding: 6px 12px;
            font-size: 13px;
            font-family: inherit;
            outline: none;
            transition: border-color 0.2s;
        }
        .chat-input:focus { border-color: #3b82f6; }
        .send-btn {
            background: #3b82f6;
            color: white;
            border: none;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background 0.2s;
        }
        .send-btn:hover { background: #2563eb; }
        .send-btn:disabled { background: #94a3b8; cursor: not-allowed; }

        .footer {
            padding: 8px 16px;
            border-top: 1px solid #e2e8f0;
            font-size: 12px;
            color: #64748b;
            background: #f8fafc;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        .upgrade-link {
            color: #2563eb;
            text-decoration: none;
            font-weight: 600;
            background: #eff6ff;
            padding: 4px 8px;
            border-radius: 4px;
            transition: background 0.2s;
        }
        .upgrade-link:hover { background: #dbeafe; }
        .loading { 
            color: #64748b; 
            font-style: italic; 
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .error { 
            color: #ef4444; 
            background: #fef2f2; 
            padding: 8px; 
            border-radius: 4px; 
            border: 1px solid #fee2e2;
            font-size: 13px; /* 显式设置字体大小 */
            line-height: 1.4;
            display: block; /* 防止被设置为 none 或其他 */
            visibility: visible;
        }

        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 200;
            animation: fadeIn 0.2s;
            pointer-events: auto; /* 允许点击 */
        }
        .modal {
            background: white;
            border-radius: 16px;
            width: 320px;
            padding: 24px;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            text-align: center;
            font-family: 'Segoe UI', system-ui, sans-serif;
            color: #334155;
            position: relative;
        }
        .modal h2 {
            margin: 0 0 16px;
            color: #1e293b;
            font-size: 20px;
        }
        .modal p {
            margin: 0 0 12px;
            line-height: 1.6;
            font-size: 14px;
            color: #475569;
            text-align: left;
        }
        .modal .highlight {
            color: #2563eb;
            font-weight: 600;
        }
        .modal-btn {
            background: linear-gradient(135deg, #3b82f6, #2563eb);
            color: white;
            border: none;
            width: 100%;
            padding: 12px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            margin-top: 16px;
            font-size: 15px;
            transition: transform 0.1s;
        }
        .modal-btn:hover {
            transform: translateY(-1px);
        }
        .modal-close {
            position: absolute;
            top: 12px;
            right: 12px;
            cursor: pointer;
            color: #94a3b8;
            background: none;
            border: none;
            font-size: 20px;
        }
        .modal-footer-quote {
            margin-top: 16px;
            font-style: italic;
            font-size: 12px;
            color: #64748b;
            border-top: 1px solid #e2e8f0;
            padding-top: 12px;
        }
    `;
  shadow.appendChild(style);

  return { host, shadow };
}

const { host, shadow } = createUI();
let currentIcon = null;
let currentPanel = null;

function isEditableTarget(target) {
  if (!target) return false;
  const tag = (target.tagName || "").toLowerCase();
  if (tag === "input" || tag === "textarea" || tag === "select") return true;
  if (target.isContentEditable) return true;
  return false;
}

function matchHotkey(e, hk) {
  if (!hk || !hk.code) return false;
  if (e.repeat) return false;
  if (e.code !== hk.code) return false;
  if (Boolean(hk.shift) !== e.shiftKey) return false;
  if (Boolean(hk.ctrl) !== e.ctrlKey) return false;
  if (Boolean(hk.alt) !== e.altKey) return false;
  if (Boolean(hk.meta) !== e.metaKey) return false;
  return true;
}

function getSelectionPoint() {
  const sel = window.getSelection();
  if (sel && sel.rangeCount > 0) {
    const range = sel.getRangeAt(0);
    const rect = range.getBoundingClientRect();
    if (rect && rect.width + rect.height > 0) {
      return {
        x: rect.left + window.scrollX,
        y: rect.bottom + window.scrollY,
      };
    }
  }
  return { x: LAST_SELECTION.x, y: LAST_SELECTION.y };
}

// ================= 核心逻辑 =================

// 1. 监听选区变化
document.addEventListener("mouseup", (e) => {
  const selection = window.getSelection();
  const text = selection.toString().trim();

  // 如果点击是在我们的 UI 内部，不清除
  if (host.contains(e.target)) return;
  if (e.composedPath().includes(host)) return; // Shadow DOM 穿透检查

  removeIcon();
  removePanel();

  if (text.length > 0) {
    LAST_SELECTION = { text, x: e.pageX, y: e.pageY };
    if (!DISABLE_FLOATING_WINDOW && !HOTKEY_MODE_ENABLED) {
      showIcon(e.pageX, e.pageY, text);
    }
  }
});

document.addEventListener(
  "keydown",
  (e) => {
    if (!HOTKEY_MODE_ENABLED) return;
    if (isEditableTarget(e.target)) return;
    if (!matchHotkey(e, HOTKEY_COMBO)) return;

    const selection = window.getSelection();
    let text = selection ? selection.toString().trim() : "";
    if (!text) text = LAST_SELECTION.text;
    if (!text) return;

    const point = getSelectionPoint();
    removeIcon();
    removePanel();
    showPanel(point.x || 0, point.y || 0, text);
  },
  true,
);

// 2. 显示图标
function showIcon(x, y, text) {
  const btn = document.createElement("div");
  btn.className = "icon-btn";
  const btnText = document.createElement("span");
  btnText.textContent = t("ask_ai");
  btn.appendChild(btnText);

  // 简单的位置计算，避免溢出视口可优化
  btn.style.left = `${x + 10}px`;
  btn.style.top = `${y + 10}px`;

  // 分离 mousedown 和 click 事件
  // mousedown 仅用于阻止默认行为（防止选区丢失）
  btn.onmousedown = (e) => {
    e.preventDefault();
    e.stopPropagation();
  };

  // click 用于触发业务逻辑，解决“必须长按”的问题
  btn.onclick = (e) => {
    e.stopPropagation();
    removeIcon();
    showPanel(x, y, text);
  };

  shadow.appendChild(btn);
  currentIcon = btn;
}

function removeIcon() {
  if (currentIcon) {
    currentIcon.remove();
    currentIcon = null;
  }
}

// 3. 显示面板（主要逻辑）
async function showPanel(x, y, text) {
  const deviceId = await getDeviceId();
  if (!deviceId) return; // 上下文丢失，终止执行

  // 获取用户偏好的 detail_level
  let detailLevel = "auto";
  try {
    const settings = await chrome.storage.local.get(["detail_level"]);
    if (settings && settings.detail_level) {
      detailLevel = settings.detail_level;
    }
  } catch (e) {
    console.warn("Failed to load settings:", e);
  }

  // 创建面板容器（Shadow DOM）
  const panel = document.createElement("div");
  panel.className = "panel";
  panel.style.left = `${x}px`;
  panel.style.top = `${y + 20}px`;

  const createSvg = (viewBox, pathD) => {
    const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
    svg.setAttribute("viewBox", viewBox);
    svg.style.width = "16px";
    svg.style.height = "16px";
    svg.style.fill = "currentColor";
    const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
    path.setAttribute("d", pathD);
    svg.appendChild(path);
    return svg;
  };

  const createTypingIndicator = () => {
    const indicator = document.createElement("div");
    indicator.className = "typing-indicator";
    for (let i = 0; i < 3; i += 1) {
      const dot = document.createElement("div");
      dot.className = "typing-dot";
      indicator.appendChild(dot);
    }
    return indicator;
  };

  const headerEl = document.createElement("div");
  headerEl.className = "header";

  const titleEl = document.createElement("span");
  titleEl.textContent = t("panel_title");

  const closeBtnEl = document.createElement("span");
  closeBtnEl.className = "close-btn";
  closeBtnEl.appendChild(
    createSvg(
      "0 0 24 24",
      "M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z",
    ),
  );
  closeBtnEl.onclick = removePanel;

  headerEl.appendChild(titleEl);
  headerEl.appendChild(closeBtnEl);

  const contentEl = document.createElement("div");
  contentEl.className = "content";

  const initialUserMsgDiv = document.createElement("div");
  initialUserMsgDiv.className = "msg-user";
  const initialUserMsgContent = document.createElement("div");
  initialUserMsgContent.className = "msg-user-content";
  initialUserMsgContent.textContent = text;
  initialUserMsgDiv.appendChild(initialUserMsgContent);

  const initialAiContainer = document.createElement("div");
  initialAiContainer.className = "msg-ai";
  initialAiContainer.appendChild(createTypingIndicator());

  contentEl.appendChild(initialUserMsgDiv);
  contentEl.appendChild(initialAiContainer);

  const inputAreaEl = document.createElement("div");
  inputAreaEl.className = "input-area";

  const input = document.createElement("input");
  input.type = "text";
  input.className = "chat-input";
  input.placeholder = t("placeholder_ask");
  input.disabled = true;

  const sendBtn = document.createElement("button");
  sendBtn.className = "send-btn";
  sendBtn.disabled = true;
  sendBtn.appendChild(
    createSvg("0 0 24 24", "M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"),
  );

  inputAreaEl.appendChild(input);
  inputAreaEl.appendChild(sendBtn);

  const footerEl = document.createElement("div");
  footerEl.className = "footer";
  footerEl.textContent = t("footer_checking");

  const typingStyle = document.createElement("style");
  typingStyle.textContent = `
        .typing-indicator {
            display: inline-flex;
            align-items: center;
            background: #f1f5f9;
            padding: 12px 16px;
            border-radius: 18px;
            border-bottom-left-radius: 4px;
            margin: 8px 0;
            width: fit-content;
        }
        .typing-dot {
            width: 6px;
            height: 6px;
            margin: 0 3px;
            background: #64748b;
            border-radius: 50%;
            animation: typing 1.4s infinite ease-in-out both;
        }
        .typing-dot:nth-child(1) { animation-delay: -0.32s; }
        .typing-dot:nth-child(2) { animation-delay: -0.16s; }
        @keyframes typing {
            0%, 80%, 100% { transform: scale(0); }
            40% { transform: scale(1); }
        }
    `;

  panel.appendChild(headerEl);
  panel.appendChild(contentEl);
  panel.appendChild(inputAreaEl);
  panel.appendChild(footerEl);
  panel.appendChild(typingStyle);

  // --- 拖拽逻辑 ---
  headerEl.style.cursor = "move";

  let isDragging = false;
  let dragOffsetX = 0;
  let dragOffsetY = 0;

  headerEl.onmousedown = (e) => {
    isDragging = true;
    // 计算鼠标相对于面板左上角的偏移量
    const rect = panel.getBoundingClientRect();
    dragOffsetX = e.clientX - rect.left;
    dragOffsetY = e.clientY - rect.top;

    // 防止选区丢失或拖拽文本
    e.preventDefault();

    const onMouseMove = (moveEvent) => {
      if (!isDragging) return;
      // 计算新的位置（文档坐标）
      // moveEvent.pageX 是鼠标在文档中的坐标
      // 我们希望 panel 的左上角移动到 (moveEvent.pageX - dragOffsetX)
      // 注意：dragOffsetX 是鼠标相对于 panel 的偏移，这个偏移是固定的
      const newX = moveEvent.pageX - dragOffsetX;
      const newY = moveEvent.pageY - dragOffsetY;

      panel.style.left = `${newX}px`;
      panel.style.top = `${newY}px`;
    };

    const onMouseUp = () => {
      isDragging = false;
      document.removeEventListener("mousemove", onMouseMove);
      document.removeEventListener("mouseup", onMouseUp);
    };

    document.addEventListener("mousemove", onMouseMove);
    document.addEventListener("mouseup", onMouseUp);
  };
  // ----------------

  shadow.appendChild(panel);
  currentPanel = panel;

  const contentDiv = contentEl;
  const footerDiv = footerEl;

  // 初始化对话历史
  let conversationHistory = [];
  conversationHistory.push({ role: "user", content: text });

  // 输入框逻辑
  async function handleSend() {
    const question = input.value.trim();
    if (!question) return;

    // 禁用输入
    input.value = "";
    input.disabled = true;
    sendBtn.disabled = true;

    // 添加用户消息到 UI
    const userMsgDiv = document.createElement("div");
    userMsgDiv.className = "msg-user";
    const userMsgContent = document.createElement("div");
    userMsgContent.className = "msg-user-content";
    userMsgContent.textContent = question;
    userMsgDiv.appendChild(userMsgContent);
    contentDiv.appendChild(userMsgDiv);

    // 添加 AI 占位符
    const aiMsgDiv = document.createElement("div");
    aiMsgDiv.className = "msg-ai";
    aiMsgDiv.appendChild(createTypingIndicator());
    contentDiv.appendChild(aiMsgDiv);
    contentDiv.scrollTop = contentDiv.scrollHeight;

    // 更新历史
    conversationHistory.push({ role: "user", content: question });

    // 发送请求
    await streamQuery(
      null,
      deviceId,
      aiMsgDiv,
      detailLevel,
      conversationHistory,
      () => {
        // 完成回调：启用输入
        input.disabled = false;
        sendBtn.disabled = false;
        input.focus();
      },
    );
  }

  sendBtn.onclick = handleSend;
  input.onkeydown = (e) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSend();
    }
  };

  // 更新底部状态栏
  updateFooter(footerDiv, deviceId);

  // 发起初始流式请求
  // 初始请求使用 text 参数（兼容旧逻辑，且 server 端已支持自动构建 history）
  // 但为了维护本地 conversationHistory，我们需要捕获它的输出
  // streamQuery 现在支持传入 messages，如果传了 messages，就用 messages。
  // 初始调用：我们传递 null 作为 text (因为已经放入 history)，传递 conversationHistory
  // 或者，为了简单，初始调用我们依然传 text，但在 streamQuery 内部处理好 history 更新？
  // 最好统一：streamQuery 负责网络请求和 UI 更新，并最后更新 history

  // 第一次调用，UI 已经有了 msg-ai 容器 (initialAiContainer)
  await streamQuery(
    null,
    deviceId,
    initialAiContainer,
    detailLevel,
    conversationHistory,
    () => {
      input.disabled = false;
      sendBtn.disabled = false;
      input.focus();
    },
  );
}

function removePanel() {
  if (currentPanel) {
    currentPanel.remove();
    currentPanel = null;
  }
}

async function showTrialEndedModal(savedMinutes, deviceId) {
  const overlay = document.createElement("div");
  overlay.className = "modal-overlay";

  const upgradeUrl = buildAfdianItemUrl(deviceId);

  const modal = document.createElement("div");
  modal.className = "modal";

  const closeBtn = document.createElement("button");
  closeBtn.className = "modal-close";
  closeBtn.type = "button";
  closeBtn.textContent = "×";
  closeBtn.onclick = () => overlay.remove();

  const title = document.createElement("h2");
  title.textContent = t("trial_modal_title");

  const desc = document.createElement("p");
  const descTemplate = t("trial_modal_desc");
  const parts = descTemplate.split("{minutes}");
  desc.appendChild(document.createTextNode(parts[0] || ""));
  const highlight = document.createElement("span");
  highlight.className = "highlight";
  const minutesNum = Number(savedMinutes);
  const minutesSafe = Number.isFinite(minutesNum)
    ? Math.max(0, Math.round(minutesNum))
    : 0;
  highlight.textContent = String(minutesSafe);
  desc.appendChild(highlight);
  desc.appendChild(document.createTextNode(parts[1] || ""));

  const info = document.createElement("p");
  info.textContent = t("trial_modal_info");

  const pro = document.createElement("p");
  pro.textContent = t("trial_modal_pro");

  const cta = document.createElement("p");
  cta.textContent = t("trial_modal_cta");

  const upgradeBtn = document.createElement("button");
  upgradeBtn.className = "modal-btn";
  upgradeBtn.type = "button";
  upgradeBtn.textContent = t("trial_modal_btn");
  upgradeBtn.onclick = () => window.open(upgradeUrl, "_blank");

  const quote = document.createElement("div");
  quote.className = "modal-footer-quote";
  quote.textContent = t("trial_modal_quote");

  modal.appendChild(closeBtn);
  modal.appendChild(title);
  modal.appendChild(desc);
  modal.appendChild(info);
  modal.appendChild(pro);
  modal.appendChild(cta);
  modal.appendChild(upgradeBtn);
  modal.appendChild(quote);

  overlay.appendChild(modal);

  // Close on click outside
  overlay.onclick = (e) => {
    if (e.target === overlay) overlay.remove();
  };

  shadow.appendChild(overlay);
}

async function updateFooter(container, deviceId) {
  const upgradeUrl = buildAfdianItemUrl(deviceId);
  const renderFooterWithLink = (prefixText, linkText) => {
    clearElement(container);
    if (prefixText) {
      container.appendChild(document.createTextNode(prefixText));
      container.appendChild(document.createTextNode(" "));
    }
    const a = document.createElement("a");
    a.href = upgradeUrl;
    a.target = "_blank";
    a.className = "upgrade-link";
    a.textContent = linkText;
    container.appendChild(a);
  };

  // 代理到 background.js 请求 (解决 Mixed Content 问题)
  try {
    const response = await new Promise((resolve) => {
      chrome.runtime.sendMessage(
        {
          type: "FETCH_USER_STATUS",
          deviceId: deviceId,
        },
        resolve,
      );
    });

    if (!response || !response.success) {
      throw new Error(response ? response.error : "Unknown error");
    }

    const data = response.data;

    // 检查是否需要显示试用结束弹窗
    // 逻辑：如果试用已结束，且不是 Pro，且没有显示过弹窗
    if (!data.is_pro) {
      if (!data.trial_active) {
        const hasSeen = await chrome.storage.local.get(["trial_ended_seen"]);
        if (!hasSeen.trial_ended_seen) {
          showTrialEndedModal(data.saved_minutes ?? 0, deviceId);
          await chrome.storage.local.set({ trial_ended_seen: true });
        }
      } else {
        await chrome.storage.local.remove(["trial_ended_seen"]);
      }
    }

    if (data.is_pro) {
      clearElement(container);
      container.textContent = t("status_pro");
    } else if (data.trial_active) {
      // 试用期显示追问次数 (X/20)
      // 使用后端返回的 followup_count，如果没有则默认为 0
      const followupCount =
        data.followup_count !== undefined && data.followup_count !== null
          ? data.followup_count
          : 0;
      renderFooterWithLink(
        t("status_trial_prefix").replace("{count}", followupCount),
        t("status_upgrade_trial"),
      );
    } else {
      // 使用后端返回的 daily_query_count 和 limit 3
      // 确保 daily_query_count 有值，避免 undefined
      const dailyCount =
        data.daily_query_count !== undefined && data.daily_query_count !== null
          ? data.daily_query_count
          : 0;
      renderFooterWithLink(
        t("status_free_prefix").replace("{count}", dailyCount),
        t("status_upgrade_free"),
      );
    }
  } catch (e) {
    console.error("Status check failed. DeviceID:", deviceId, "Error:", e);
    renderFooterWithLink(
      t("status_offline_prefix"),
      t("status_upgrade_offline"),
    );
  }
}

// 4. SSE 流式处理
async function streamQuery(
  text,
  deviceId,
  container,
  detailLevel,
  conversationHistory,
  onComplete,
) {
  let fullText = "";
  let isFirstChunk = true;
  let buffer = "";
  let contentEl = null;
  let renderScheduled = false;
  let isComplete = false;

  const safeOnComplete = () => {
    if (!isComplete) {
      isComplete = true;
      if (onComplete) onComplete();
    }
  };

  const renderError = (message, hint) => {
    clearElement(container);
    const err = document.createElement("div");
    err.className = "error";
    const strong = document.createElement("strong");
    strong.textContent = "Error:";
    err.appendChild(strong);
    err.appendChild(document.createTextNode(` ${message}`));
    if (hint) {
      err.appendChild(document.createElement("br"));
      const span = document.createElement("span");
      span.style.fontSize = "0.9em";
      span.style.opacity = "0.8";
      span.textContent = hint;
      err.appendChild(span);
    }
    container.appendChild(err);
  };

  const scheduleRender = () => {
    if (!contentEl || renderScheduled) return;
    renderScheduled = true;
    requestAnimationFrame(() => {
      renderScheduled = false;
      if (!contentEl) return;
      renderMarkdown(contentEl, fullText);
      const panelContent = container.closest(".content");
      if (panelContent) panelContent.scrollTop = panelContent.scrollHeight;
    });
  };

  try {
    // 获取 AI 模型设置
    const { ai_model } = await chrome.storage.local.get(["ai_model"]);

    // 使用 Port 连接 background.js 进行流式请求 (解决 Mixed Content 问题)
    const port = chrome.runtime.connect({ name: "STREAM_QUERY" });

    // 监听意外断开
    port.onDisconnect.addListener(() => {
      if (!isComplete) {
        const lastError = chrome.runtime.lastError;
        console.warn("Stream disconnected unexpectedly:", lastError);
        // 如果没有收到内容且有错误，显示错误
        if (!fullText && lastError) {
          renderError(t("error_network"), lastError.message);
        }
        safeOnComplete();
      }
    });

    port.onMessage.addListener((msg) => {
      if (msg.type === "ERROR") {
        const errorMsg = msg.error || t("error_unknown");
        const hint = errorMsg.includes("HTTP Error") ? t("error_network") : "";
        renderError(errorMsg, hint);

        port.disconnect();
        safeOnComplete();
        return;
      }

      if (msg.type === "DONE") {
        // 完成后，更新 history 的最后一条（即 AI 的回复）
        if (conversationHistory) {
          conversationHistory.push({ role: "assistant", content: fullText });
        }
        // 立即刷新底部状态栏 (追问次数/每日限制)
        // Fix: footerEl is not defined in this scope, look it up from currentPanel
        const footerEl = currentPanel
          ? currentPanel.querySelector(".footer")
          : null;
        if (footerEl) {
          updateFooter(footerEl, deviceId);
        }
        port.disconnect();
        safeOnComplete();
        return;
      }

      if (msg.type === "CHUNK") {
        const chunk = msg.chunk;
        buffer += chunk;

        // 处理 SSE 格式 (data: ...)
        const lines = buffer.split("\n\n");
        buffer = lines.pop(); // 保留未完成的部分

        for (const line of lines) {
          if (line.startsWith("data: ")) {
            const dataStr = line.slice(6);
            try {
              if (dataStr.trim() === "[DONE]") continue;

              const json = JSON.parse(dataStr);

              // 处理后端返回的特定错误
              if (json.error) {
                const errorMsg = json.message || t("error_unknown");
                const businessErrors = [
                  "LIMIT_REACHED",
                  "TRIAL_LIMIT_REACHED",
                  "PRO_REQUIRED",
                  "DAILY_LIMIT_REACHED",
                  "RATE_LIMIT_EXCEEDED",
                ];
                const hint = businessErrors.includes(json.error)
                  ? ""
                  : t("error_check_network");
                renderError(errorMsg, hint);
                port.disconnect();
                safeOnComplete();
                return;
              }

              // 处理 OpenAI 兼容格式
              const content =
                json.choices?.[0]?.delta?.content || json.content || "";

              if (content) {
                if (isFirstChunk) {
                  clearElement(container);
                  contentEl = document.createElement("div");
                  contentEl.className = "msg-ai-content";
                  container.appendChild(contentEl);
                  isFirstChunk = false;
                }

                fullText += content;
                scheduleRender();
              }
            } catch (e) {
              console.error("Parse error:", e);
            }
          }
        }
      }
    });

    // 发送请求参数
    port.postMessage({
      type: "START_QUERY",
      payload: {
        device_id: deviceId,
        text: text,
        messages: conversationHistory,
        detail_level: detailLevel,
        language: CURRENT_LANG,
        model: ai_model,
      },
    });
  } catch (e) {
    console.error("Stream initialization failed:", e);
    renderError(t("error_unknown"), e.message);
    safeOnComplete();
  }
}
