// background.js

const API_BASE = "https://api.mtvoc.com";

chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === "install") {
    // Install: Open the options page for onboarding
    chrome.runtime.openOptionsPage();
  } else if (details.reason === "update") {
    // Update: Do nothing or notify (optional)
  }
});

chrome.action.onClicked.addListener(() => {
  chrome.runtime.openOptionsPage();
});

// 1. 处理普通请求 (如 user/status)
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.type === "FETCH_USER_STATUS") {
    const deviceId = encodeURIComponent(request.deviceId || "");
    const url = `${API_BASE}/user/status?device_id=${deviceId}&_=${Date.now()}`;
    fetch(url)
      .then((res) => {
        if (!res.ok) {
          throw new Error(`HTTP Error ${res.status}`);
        }
        return res.json();
      })
      .then((data) => sendResponse({ success: true, data }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true; // Keep channel open for async response
  }
});

// 2. 处理流式请求 (Query)
chrome.runtime.onConnect.addListener((port) => {
  if (port.name === "STREAM_QUERY") {
    port.onMessage.addListener(async (msg) => {
      if (msg.type === "START_QUERY") {
        try {
          const response = await fetch(`${API_BASE}/query`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(msg.payload),
          });

          if (!response.ok) {
            port.postMessage({
              type: "ERROR",
              error: `HTTP Error: ${response.status}`,
            });
            return;
          }

          const reader = response.body.getReader();
          const decoder = new TextDecoder();

          while (true) {
            const { done, value } = await reader.read();
            if (done) break;
            const chunk = decoder.decode(value, { stream: true });
            port.postMessage({ type: "CHUNK", chunk });
          }

          port.postMessage({ type: "DONE" });
        } catch (error) {
          port.postMessage({ type: "ERROR", error: error.message });
        }
      }
    });
  }
});
